// SPDX-License-Identifier: GPL-2.0-only

#pragma once

#include <stdint.h>

/*
 * Low-level I/O routines.
 *
 * Copied from <file:arch/powerpc/include/asm/io.h> (which has no copyright)
 */
static inline uint8_t in_8(const volatile uint8_t *addr)
{
	int ret;

	__asm__ __volatile__("lbz%U1%X1 %0,%1; twi 0,%0,0; isync"
			     : "=r" (ret) : "m" (*addr));
	return ret;
}

static inline void out_8(volatile uint8_t *addr, uint8_t val)
{
	__asm__ __volatile__("stb%U0%X0 %1,%0; sync"
			     : "=m" (*addr) : "r" (val));
}

static inline uint16_t in_le16(const volatile uint16_t *addr)
{
	uint32_t ret;

	__asm__ __volatile__("lhbrx %0,0,%1; twi 0,%0,0; isync"
			     : "=r" (ret) : "r" (addr), "m" (*addr));

	return ret;
}

static inline uint16_t in_be16(const volatile uint16_t *addr)
{
	uint32_t ret;

	__asm__ __volatile__("lhz%U1%X1 %0,%1; twi 0,%0,0; isync"
			     : "=r" (ret) : "m" (*addr));
	return ret;
}

static inline void out_le16(volatile uint16_t *addr, uint16_t val)
{
	__asm__ __volatile__("sthbrx %1,0,%2; sync" : "=m" (*addr)
			     : "r" (val), "r" (addr));
}

static inline void out_be16(volatile uint16_t *addr, uint16_t val)
{
	__asm__ __volatile__("sth%U0%X0 %1,%0; sync"
			     : "=m" (*addr) : "r" (val));
}

static inline uint32_t in_le32(const volatile uint32_t *addr)
{
	uint32_t ret;

	__asm__ __volatile__("lwbrx %0,0,%1; twi 0,%0,0; isync"
			     : "=r" (ret) : "r" (addr), "m" (*addr));
	return ret;
}

static inline uint32_t in_be32(const volatile uint32_t *addr)
{
	uint32_t ret;

	__asm__ __volatile__("lwz%U1%X1 %0,%1; twi 0,%0,0; isync"
			     : "=r" (ret) : "m" (*addr));
	return ret;
}

static inline void out_le32(volatile uint32_t *addr, uint32_t val)
{
	__asm__ __volatile__("stwbrx %1,0,%2; sync" : "=m" (*addr)
			     : "r" (val), "r" (addr));
}

static inline void out_be32(volatile uint32_t *addr, uint32_t val)
{
	__asm__ __volatile__("stw%U0%X0 %1,%0; sync"
			     : "=m" (*addr) : "r" (val));
}

static inline void sync(void)
{
	asm volatile("sync" : : : "memory");
}

static inline void eieio(void)
{
	asm volatile("eieio" : : : "memory");
}

static inline void barrier(void)
{
	asm volatile("" : : : "memory");
}
