#!/usr/bin/env python

""" Update the ``ask_generated.py`` file.

This must be run each time ``known_facts()`` in ``assumptions.facts`` module
is changed.

Should be run from sympy root directory.

$ python bin/ask_update.py
"""

# hook in-tree SymPy into Python path, if possible
import os
import sys

isympy_path = os.path.abspath(__file__)
isympy_dir = os.path.dirname(isympy_path)
sympy_top = os.path.split(isympy_dir)[0]
sympy_dir = os.path.join(sympy_top, 'sympy')

if os.path.isdir(sympy_dir):
    sys.path.insert(0, sympy_top)

from sympy.assumptions.cnf import CNF, Literal
from sympy.assumptions.facts import (get_known_facts,
    generate_known_facts_dict, get_known_facts_keys)
from sympy.core import Symbol

def generate_code():
    from textwrap import dedent, wrap

    LINE = ",\n        "
    HANG = ' '*8
    code_string = dedent('''\
    """
    Do NOT manually edit this file.
    Instead, run ./bin/ask_update.py.
    """

    from sympy.assumptions.ask import Q
    from sympy.assumptions.cnf import Literal
    from sympy.core.cache import cacheit

    @cacheit
    def get_all_known_facts():
        """
        Known facts between unary predicates as CNF clauses.
        """
        return {
            %s
        }

    @cacheit
    def get_known_facts_dict():
        """
        Logical relations between unary predicates as dictionary.

        Each key is a predicate, and item is two groups of predicates.
        First group contains the predicates which are implied by the key, and
        second group contains the predicates which are rejected by the key.

        """
        return {
            %s
        }
    ''')

    x = Symbol('x')
    fact = get_known_facts(x)

    # Generate CNF of facts between known unary predicates
    cnf = CNF.to_CNF(fact)
    p = LINE.join(sorted([
        'frozenset(('
         + ', '.join(str(Literal(lit.arg.function, lit.is_Not))
                     for lit in sorted(clause, key=str))
        + '))' for clause in cnf.clauses]))

    # Generate dictionary of facts between known unary predicates
    keys = [pred(x) for pred in get_known_facts_keys()]
    mapping = generate_known_facts_dict(keys, fact)
    items = sorted(mapping.items(), key=str)
    keys = [str(i[0]) for i in items]
    values = ['(set(%s), set(%s))' % (sorted(i[1][0], key=str),
                                      sorted(i[1][1], key=str))
              for i in items]
    m = LINE.join(['\n'.join(
        wrap("{}: {}".format(k, v),
            subsequent_indent=HANG,
            break_long_words=False))
        for k, v in zip(keys, values)]) + ','

    return code_string % (p, m)

with open('sympy/assumptions/ask_generated.py', 'w') as f:
    code = generate_code()
    f.write(code)
