/*
  ==============================================================================

   This file is part of the JUCE framework.
   Copyright (c) Raw Material Software Limited

   JUCE is an open source framework subject to commercial or open source
   licensing.

   By downloading, installing, or using the JUCE framework, or combining the
   JUCE framework with any other source code, object code, content or any other
   copyrightable work, you agree to the terms of the JUCE End User Licence
   Agreement, and all incorporated terms including the JUCE Privacy Policy and
   the JUCE Website Terms of Service, as applicable, which will bind you. If you
   do not agree to the terms of these agreements, we will not license the JUCE
   framework to you, and you must discontinue the installation or download
   process and cease use of the JUCE framework.

   JUCE End User Licence Agreement: https://juce.com/legal/juce-8-licence/
   JUCE Privacy Policy: https://juce.com/juce-privacy-policy
   JUCE Website Terms of Service: https://juce.com/juce-website-terms-of-service/

   Or:

   You may also use this code under the terms of the AGPLv3:
   https://www.gnu.org/licenses/agpl-3.0.en.html

   THE JUCE FRAMEWORK IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL
   WARRANTIES, WHETHER EXPRESSED OR IMPLIED, INCLUDING WARRANTY OF
   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, ARE DISCLAIMED.

  ==============================================================================
*/

/* This file was generated with JUCE-Utils/Unicode/generate_unicode_blob.py */
namespace juce
{
enum class LineBreakType : uint8_t
{
    al, bk, cm, cr, gl, lf, nl, sp, wj, zw, zwj, ai, b2, ba, bb, cb, cj, cl, cp, eb, em, ex, h2, h3, hl, hy, in, is, jl, id, jt, jv, ns, nu, op, po, pr, qu, ri, sa, sg, sy, xx, opw
};

enum class EastAsianWidthType : uint8_t
{
    neutral, narrow, ambiguous, full, half, wide
};

enum class VerticalTransformType : uint8_t
{
    r, u, tr, tu
};

enum class EmojiType : uint8_t
{
    yes, presentation, modifier, modifierBase, component, extended, no
};

enum class BreakOpportunity : uint8_t
{
    direct, indirect, prohibited, combinedIndirect, combinedProhibited
};

struct UnicodeEntry
{
    LineBreakType         bt;
    EastAsianWidthType    asian;
    VerticalTransformType vertical;
    EmojiType             emoji;
};

class UnicodeDataTable
{
    static auto computeHash()
    {
        uint32_t v = 5381;

        for (auto c : compressedData)
            v = ((v << 5) + v) + (uint32_t) c;

        return v;
    }

public:
    UnicodeDataTable() = delete;

    static UnicodeEntry getDataForCodepoint (uint32_t codepoint)
    {
        static const auto data = []
        {
            jassert (computeHash() == 0x58aac9b4);

            MemoryInputStream           mStream { compressedData, std::size (compressedData), false };
            GZIPDecompressorInputStream zStream { &mStream, false };

            Array<UnicodeEntry> arr;
            arr.resize (uncompressedSize / sizeof (UnicodeEntry));
            [[maybe_unused]] const auto r = zStream.read (arr.getRawDataPointer(), uncompressedSize);
            jassert (r == uncompressedSize);

            return arr;
        }();

        jassert (isPositiveAndBelow (codepoint, data.size()));
        return data[(int) codepoint];
    }

private:
    static constexpr auto    uncompressedSize = 4456444;
    static constexpr uint8_t compressedData[]
    {
        0x78,0x9c,0xec,0xdd,0x0b,0x77,0xa3,0x48,0x7a,0x00,0x50,0xa4,0x36,0x9d,0x4d,0x7a,0xed,0xdd,0xce,0xc4,0xc9,0x66,0x66,0xbd,
        0xeb,0xb1,0xdd,0x99,0x44,0xff,0xff,0x07,0x46,0x58,0xd0,0xc6,0x65,0x1e,0xc5,0xb3,0x40,0xbe,0xc7,0xe7,0x1e,0x49,0x08,0xf8,
        0xea,0x4d,0x81,0xb0,0x74,0xcc,0xb2,0xaf,0xc7,0x1e,0xb7,0x67,0xf9,0xd9,0xa1,0xf4,0x25,0x62,0x9b,0x58,0xff,0x72,0xc8,0xbe,
        0xde,0x9f,0xfd,0x38,0xcb,0x0e,0xd9,0xcd,0xcb,0xf9,0xf1,0xf9,0xf2,0xfc,0x75,0xd9,0xd3,0xd9,0xbf,0xd7,0xde,0xfb,0xed,0xec,
        0xbf,0xcb,0xc7,0xff,0x3b,0xfb,0xfd,0xbc,0x3c,0xc6,0x6f,0xe5,0x36,0x59,0xcd,0x7d,0xf0,0x7a,0x69,0x45,0x5e,0x5e,0xde,0xf2,
        0xb3,0xba,0x22,0xfe,0xed,0xd9,0xf7,0xf2,0x75,0x5b,0x9d,0x7c,0x9d,0xb1,0x7e,0xfb,0xdc,0x9c,0x3d,0x1d,0x2f,0x75,0x5e,0x94,
        0xcd,0xcb,0xf1,0xf2,0x58,0xa4,0xef,0xdb,0xf1,0x70,0x76,0x7e,0x9e,0x1d,0xf2,0xe2,0xf1,0x47,0x76,0x59,0x7e,0x5b,0x2c,0x3b,
        0x1e,0xf2,0xe2,0xf9,0x73,0xb1,0x7e,0xb9,0x5e,0xe1,0xee,0x75,0xfd,0xec,0xe7,0xeb,0xba,0x1f,0x59,0xb5,0xcf,0x8b,0xa7,0x72,
        0xdd,0x46,0x5d,0xef,0x45,0xae,0x57,0xc4,0x68,0x7d,0xff,0x18,0x68,0xdb,0x6f,0xb8,0x4e,0xb8,0x7e,0xdb,0xf2,0x7a,0xfc,0x70,
        0xfb,0x30,0xdd,0xb1,0x79,0xed,0x2b,0x83,0xbe,0xfd,0x0c,0x89,0xd3,0x54,0x2e,0x31,0x65,0x39,0x24,0x66,0x53,0x1d,0x34,0xc5,
        0xec,0x5b,0xa7,0x2d,0xbd,0x53,0xcb,0x34,0x36,0xff,0xac,0xaf,0xad,0x3f,0xc5,0x3e,0x72,0x3d,0xe6,0x1a,0x3f,0x49,0xaf,0x56,
        0x17,0xaf,0xf3,0x89,0xec,0xfd,0x1c,0xa2,0x7a,0x5d,0x5f,0xa7,0x6b,0x7f,0xe1,0x1c,0xa4,0xbe,0xcd,0xdd,0xa0,0x7a,0x3f,0x94,
        0xe6,0xcd,0xef,0xf1,0xc8,0x58,0x37,0x23,0xb7,0x69,0x33,0x26,0x0d,0x7d,0xf5,0x7b,0x2a,0x85,0xcb,0x7f,0x0b,0xde,0x6f,0x5b,
        0x2f,0xdc,0x57,0xd6,0xb4,0xde,0x71,0xba,0x53,0xdf,0x7e,0xfa,0xfa,0xec,0x54,0x53,0xe2,0xef,0xdd,0x94,0xfc,0xcd,0x51,0xf6,
        0x7b,0x36,0xb5,0xfc,0x36,0xa4,0xeb,0x3c,0x3d,0x75,0xda,0xa0,0x49,0xdf,0x31,0x2b,0x75,0xda,0xb6,0x9c,0xbe,0x58,0xc5,0x5c,
        0xe1,0xb6,0x21,0x3f,0x2f,0xe5,0xeb,0xb5,0xae,0x15,0xce,0xe9,0x36,0x7b,0x1b,0xd7,0x8e,0x0d,0xcf,0xef,0xcb,0xc7,0x70,0x7e,
        0x14,0xfa,0xdb,0x8a,0xfa,0x62,0x87,0xed,0x2e,0x46,0x4c,0xfd,0x3f,0xd7,0xfc,0x96,0xbd,0xcd,0x1d,0xeb,0xe5,0x15,0xe3,0xbe,
        0xf6,0x58,0x49,0xdd,0xb6,0x87,0x58,0xa2,0x1d,0xfe,0x1e,0xe9,0xb9,0xf6,0x3c,0x4c,0x53,0xea,0x72,0xe1,0x4d,0xd5,0xa6,0x63,
        0xe6,0x52,0xb1,0xef,0x37,0xad,0x5b,0xad,0x13,0xdb,0x7e,0xe6,0xce,0x67,0x7d,0xec,0x48,0xd9,0x06,0xd7,0x3c,0x66,0x5c,0xcb,
        0xf1,0x7c,0x0f,0x86,0xd4,0x4b,0xd5,0x1e,0xc7,0x48,0xd5,0x7f,0x96,0x2a,0x93,0x70,0x9b,0xe2,0x58,0x5d,0x8d,0x49,0x55,0x9e,
        0x8b,0xf5,0x5e,0x4a,0x6b,0xa5,0x75,0x48,0xfa,0x63,0xb6,0x9b,0xa3,0x2f,0xae,0xd5,0x97,0xdb,0xd2,0x3d,0x34,0xfe,0xd8,0xf9,
        0xdb,0x52,0x4e,0xc1,0xf3,0x26,0x43,0xdb,0xeb,0x56,0x2d,0x7d,0x6c,0x59,0x23,0xc6,0x98,0xf1,0xe3,0x5a,0x2c,0x55,0xd6,0x7d,
        0xfb,0x1a,0x9a,0xbe,0xdb,0xd2,0x9a,0xc7,0xa5,0x70,0x7c,0xea,0x5b,0x3f,0x1c,0x7b,0x96,0x1e,0x8b,0x86,0x7c,0x2e,0xd1,0x96,
        0x96,0xfa,0x78,0xd4,0x57,0x67,0xe1,0xd8,0xd5,0x34,0x96,0x85,0xf1,0xda,0xc6,0xbe,0xb6,0xf1,0x3a,0x4c,0x63,0x18,0x6b,0x68,
        0xfd,0x57,0xe7,0xe7,0x4d,0xe5,0x53,0x2d,0x7f,0x09,0xe2,0x35,0xe5,0xab,0xab,0xac,0xbb,0xf2,0xb2,0x85,0xfa,0xef,0x7b,0x5e,
        0xaf,0x93,0xae,0x7a,0x8f,0xa9,0xff,0xbe,0x7a,0xee,0x6b,0x8b,0x7d,0xed,0x27,0xb6,0xfe,0x9b,0xc6,0x99,0xd8,0xf6,0x39,0xb4,
        0xfe,0xbb,0xea,0x64,0x8b,0xf5,0x3f,0xb6,0xff,0x37,0xb5,0x91,0xb6,0xfa,0x1f,0x52,0xce,0x5d,0x6d,0x63,0x4a,0xff,0x7f,0x89,
        0x88,0x31,0x24,0x9f,0x31,0xf5,0xb1,0x87,0xfe,0x3f,0xb6,0xfe,0x87,0xf6,0xff,0xbe,0xfe,0xb4,0xc6,0xf8,0xdf,0x57,0x57,0x31,
        0x63,0x40,0x5f,0x59,0xb7,0x8d,0x69,0x6d,0x63,0x5c,0x5b,0x9e,0x9b,0xda,0x4e,0xdf,0xb2,0xae,0xe5,0xb1,0x6d,0xa8,0xad,0x8e,
        0xda,0xc6,0xf4,0xb6,0x3e,0x12,0xdb,0xff,0xfb,0xda,0x49,0x97,0x39,0xe7,0x7f,0xd5,0x71,0xbf,0xaf,0xad,0xc6,0xce,0x59,0xd7,
        0x1c,0xfb,0xe7,0xda,0xff,0xd0,0xfe,0x1f,0x5b,0xff,0x5d,0x65,0xda,0xd6,0x7e,0xc3,0xe7,0x53,0xfa,0x7f,0x57,0x3a,0xee,0x22,
        0xcb,0xa6,0x9e,0xaf,0x98,0x6d,0xb6,0x5c,0xff,0x73,0x8d,0xff,0x73,0xd5,0x7f,0x93,0xa6,0xb4,0x4e,0xa9,0xff,0x30,0x8f,0x43,
        0x0c,0xed,0xfb,0x29,0xea,0x7f,0x2e,0xf5,0x3c,0x4e,0xad,0xff,0xb6,0x32,0x0f,0xe3,0x0d,0x4d,0xdb,0x12,0xe3,0x7f,0xdb,0x79,
        0x61,0x95,0xfe,0xa1,0xf3,0xbf,0x21,0x7d,0x71,0xec,0xf1,0x7a,0x4c,0xac,0x21,0xeb,0xc5,0xa4,0xab,0xef,0x58,0xde,0xd7,0x7e,
        0xba,0xd6,0xe9,0x1b,0x23,0x86,0x8e,0xff,0xf5,0x76,0x39,0x74,0x0c,0xf8,0xe3,0x93,0x08,0xf3,0xfd,0x32,0x71,0x7f,0xd9,0x80,
        0x7a,0xaa,0xae,0xeb,0x8d,0x19,0xa3,0x97,0x56,0x2f,0x9f,0x3f,0x82,0x65,0x61,0xf9,0xad,0x51,0x47,0x6b,0xc4,0xea,0x6a,0xfb,
        0x31,0x69,0x18,0xdb,0x4f,0xfb,0xda,0xe4,0xda,0x8a,0x36,0x7c,0x17,0xa8,0x2f,0xbb,0xa9,0x3d,0xbf,0x2d,0x5f,0xdf,0x37,0xb8,
        0x29,0xb7,0x0b,0xef,0xe9,0xe9,0x9b,0x57,0xcc,0x7d,0x9d,0x3b,0xbc,0xaf,0xab,0x7a,0x7c,0x3a,0xfb,0x5e,0x7b,0x1c,0x7b,0xae,
        0xb3,0x05,0xb1,0xc7,0xa1,0x3e,0xb7,0x11,0xcb,0x9b,0xe2,0x0f,0x89,0x31,0x25,0x7d,0x4b,0x39,0x65,0x6f,0xe3,0x71,0x5b,0x19,
        0x77,0xb5,0x8d,0x7a,0x9b,0xa8,0xf7,0x8d,0xb6,0x73,0xa7,0x9b,0x52,0xea,0xbe,0xde,0x35,0xf6,0x7f,0x56,0x43,0x8f,0xdf,0x61,
        0xdd,0xa6,0x4e,0x7f,0x6a,0xb1,0xe5,0xd7,0xb6,0x7d,0xea,0xb1,0xb4,0xaf,0x8f,0xb7,0x1d,0x33,0xeb,0xcf,0xf7,0xec,0xef,0xf9,
        0x81,0x84,0xfe,0x99,0xd1,0xe5,0x1f,0x2c,0x2a,0xf5,0xf8,0xb3,0x07,0x4d,0xe3,0x7c,0xcc,0xf8,0x7f,0x9a,0xb8,0xfd,0x56,0x6c,
        0x3d,0xfd,0x5b,0xad,0x9f,0xad,0x94,0xcf,0x56,0xcb,0xff,0xb3,0x69,0x3a,0x5f,0x2f,0x96,0x77,0x9d,0x87,0x2e,0x99,0x8e,0x35,
        0xeb,0xa7,0xeb,0xfa,0x13,0xcb,0x95,0x77,0xdf,0xfb,0x97,0xb6,0x77,0x00,0x00,0xe0,0x93,0x58,0xf3,0xdc,0xa3,0xfa,0xec,0xd5,
        0xfc,0x7f,0x9c,0xdb,0xac,0xff,0x73,0xcb,0xae,0xb9,0xfe,0x92,0xe7,0x5f,0xb1,0xf7,0x35,0x2d,0x55,0xf7,0xe1,0xe7,0xd7,0xb1,
        0xf7,0xda,0xcc,0x1d,0x7f,0xe8,0xe7,0xaf,0x63,0x62,0x85,0xdb,0x9d,0x26,0xc4,0x4f,0xfd,0xf9,0xe1,0xb5,0xa9,0xda,0xde,0x63,
        0xed,0x75,0xd5,0x77,0xeb,0xf7,0x9b,0x55,0xe5,0x3f,0xf5,0x3e,0xef,0xa9,0x63,0x40,0xf5,0x7e,0x75,0x1f,0x4f,0x95,0xfe,0xbb,
        0x60,0x79,0xd8,0xc7,0x6e,0xca,0xc7,0xb5,0xd2,0xcf,0x78,0x43,0xc7,0x9f,0xd8,0x7b,0xa3,0xd6,0xd2,0x75,0x5f,0x6b,0xea,0xf9,
        0xd3,0x56,0x9d,0x22,0xa5,0x6e,0x97,0x63,0xee,0xdd,0xaa,0x9b,0xba,0x7d,0xd8,0xae,0xaa,0xf1,0x6e,0xea,0x7d,0x2d,0x4b,0xeb,
        0x3a,0x7e,0x5f,0xf3,0xf1,0x3f,0x55,0xfa,0xdb,0xca,0x6f,0x48,0x3b,0x69,0x4a,0x77,0xea,0xb1,0x75,0xe9,0x71,0xbb,0xad,0xff,
        0x55,0xeb,0xa4,0x6e,0x4f,0xa9,0xdb,0x72,0xca,0x3e,0x34,0xc7,0xfc,0x6d,0xea,0xf6,0x73,0xe4,0x21,0xe5,0x7d,0xc4,0x7b,0xd7,
        0x94,0xaf,0xd4,0xe3,0xc6,0xdc,0x63,0xce,0x50,0x31,0xf3,0x96,0xa1,0xed,0xbf,0x7e,0xdd,0xe6,0x36,0x1b,0x77,0x2d,0x67,0x8e,
        0xb4,0xd7,0xe3,0x87,0xf5,0xbf,0xa7,0x3a,0xab,0xb6,0x9d,0xfb,0xff,0x47,0xb6,0x62,0xa9,0x31,0x29,0x75,0xbe,0x96,0xca,0x77,
        0x4c,0x79,0x84,0xfd,0x6e,0xc8,0xf7,0x6c,0xb5,0x95,0xe1,0x5e,0xdb,0xdf,0x90,0xb1,0x67,0xcf,0xed,0xa9,0x29,0x1f,0x63,0xb6,
        0x89,0x39,0x7e,0x2e,0xd1,0x7e,0x9b,0xc6,0xe5,0xae,0xb1,0xfa,0x18,0x3c,0xaf,0xef,0x63,0xef,0xf5,0x07,0x73,0x5b,0xa2,0xbf,
        0xee,0x49,0xea,0xf2,0x07,0xae,0x4f,0xdf,0xfc,0x62,0xeb,0xf3,0x8f,0xb9,0xd2,0x7f,0xea,0x79,0xdc,0x6a,0xfe,0xf6,0x6c,0x6a,
        0xbe,0xa6,0x6c,0x1f,0xd3,0x6e,0x96,0xae,0xc7,0xbe,0x18,0x77,0xb5,0xf5,0x9a,0xce,0x89,0x2b,0x37,0xc1,0xeb,0x7f,0xcd,0x2e,
        0x73,0x86,0x7f,0xcb,0xb2,0x9b,0x6a,0xfe,0x50,0xfc,0x16,0xfb,0xcf,0xf5,0xce,0xcf,0xff,0x7c,0xac,0x7e,0xeb,0xf4,0xf2,0xd9,
        0xe2,0x8f,0xe3,0xc5,0x53,0x76,0xf9,0xfd,0xf5,0xf0,0x75,0xf5,0x5b,0xec,0xd5,0x6f,0xe8,0xfd,0x7a,0x7c,0x53,0xec,0xef,0x90,
        0x5d,0x34,0xcd,0x5d,0x8a,0xb8,0xcf,0xe7,0x6d,0x9f,0xb3,0xc3,0xeb,0xef,0xc0,0xbf,0xca,0x6a,0x8f,0x65,0x3e,0x7f,0xd4,0x62,
        0x3d,0x66,0x87,0xfc,0xb1,0x58,0xfe,0xe1,0x37,0xfb,0x2e,0xe9,0x2d,0x7e,0x2f,0xa1,0x48,0x5b,0x71,0x9f,0xda,0x79,0xdd,0x4a,
        0xde,0x5d,0x0f,0x1f,0x3f,0x47,0xed,0xba,0xd6,0x57,0x7f,0xfd,0xa7,0xc6,0x3a,0x3d,0x0c,0x9e,0xbf,0x55,0xed,0xa0,0xef,0xf7,
        0x68,0xb3,0x5a,0xfe,0xea,0xbf,0x47,0x1b,0xfe,0x46,0x6d,0xcc,0xf6,0x63,0xdb,0x72,0xfd,0xfc,0xf3,0xa5,0xc3,0x73,0xf5,0xfc,
        0xa6,0xb6,0xfc,0xd8,0xbd,0xcd,0x87,0x6d,0x5b,0x96,0x3d,0x77,0xac,0x37,0xc4,0xb8,0x79,0xf7,0x21,0xf0,0x73,0xd9,0xcd,0x71,
        0x40,0xdd,0x57,0xa6,0x5c,0xff,0x7f,0xbb,0x67,0xe0,0xd2,0x97,0x8a,0xe7,0xdf,0x8e,0xef,0xef,0x25,0x78,0x3e,0xb6,0xdd,0x27,
        0xf0,0x7e,0xf9,0xdb,0x76,0x45,0x3d,0x1d,0x3e,0xbc,0xff,0x71,0xdb,0xb7,0x78,0x67,0xf9,0xdb,0xfa,0xd5,0x78,0x70,0x78,0x17,
        0xe7,0x7d,0xdb,0x8c,0xdb,0x77,0x29,0x6f,0xbf,0x57,0x22,0x6a,0xfb,0x20,0x2d,0xe1,0xf3,0x52,0x39,0x96,0x7d,0x6b,0xd9,0xfe,
        0x32,0xce,0xbd,0xad,0x57,0xcb,0x7b,0xb4,0xa6,0x7d,0x8e,0xd9,0x6e,0x88,0x6f,0xb5,0xfa,0x0f,0xef,0x1f,0x09,0xc7,0x8e,0xb3,
        0xbc,0xc9,0xc4,0xe3,0x5c,0x5e,0x1a,0xbe,0xed,0x4a,0xbf,0xdf,0x1b,0xfe,0xb6,0xf7,0x2c,0x56,0x4e,0x77,0x98,0xf6,0x70,0x59,
        0xd3,0xfb,0xd5,0x63,0xfd,0xf7,0x84,0x9a,0xca,0x20,0xdc,0xcf,0xa5,0x5d,0x36,0xef,0xbf,0xed,0xb8,0xd4,0xf4,0x1b,0xea,0x1f,
        0xe3,0x1c,0x1a,0xf7,0xdd,0x96,0xa7,0xd8,0xf2,0x69,0x7b,0xdd,0xb7,0x3c,0x66,0xff,0x1f,0xcb,0xa6,0x7d,0xf9,0xd8,0x7c,0x8c,
        0x4d,0xc7,0x94,0x36,0x35,0xb6,0x6c,0x66,0xed,0x43,0x1b,0x88,0xb3,0x94,0x5f,0x67,0xd9,0x4f,0xf7,0xd8,0x5f,0xcd,0xf5,0xaa,
        0xc7,0xee,0x63,0x44,0xbb,0x87,0xfc,0x90,0x17,0xfa,0x8f,0xbf,0x1f,0x8f,0xdd,0x4f,0xf9,0xf1,0xeb,0xf7,0xb3,0xee,0x63,0x3e,
        0x73,0xd5,0x79,0xfc,0xdc,0x69,0xa8,0xd4,0xf9,0x1e,0x5d,0x06,0x2d,0x79,0xa9,0xce,0x0b,0xe7,0x28,0x83,0x8e,0x6d,0x8a,0x7e,
        0x53,0x97,0xbd,0x79,0xed,0x57,0x59,0xa9,0xb1,0x7f,0xbd,0x5b,0x7f,0xa6,0x7a,0x5c,0xca,0xfb,0x72,0x88,0xbd,0x87,0x79,0x8a,
        0x21,0xe9,0x5b,0x22,0xfe,0xd0,0xf9,0x3f,0x8b,0xc9,0x37,0x90,0x86,0x76,0x0b,0xb4,0x95,0xcb,0x79,0x68,0xfb,0x3c,0x9b,0xf6,
        0xf9,0x78,0x25,0x75,0x7a,0x3e,0x18,0x78,0x1c,0x9c,0xab,0x3c,0xfa,0xca,0xa9,0x3d,0x7e,0xd9,0x16,0x7b,0xda,0x77,0xdf,0x31,
        0xac,0xbe,0x9f,0xf2,0x75,0x5e,0x7f,0x3d,0x76,0xbd,0x60,0xdd,0xc6,0xf7,0x9b,0xb6,0x1f,0x6a,0xcc,0xfe,0x46,0xc4,0xbd,0x94,
        0xe1,0xdb,0x3c,0xe2,0xeb,0xcf,0xb9,0x43,0xa9,0x78,0xbf,0x56,0x26,0x3f,0x5f,0x07,0xf3,0x88,0x6a,0x9d,0x70,0xf9,0xcf,0x79,
        0xfe,0xb7,0xec,0xa2,0xda,0xe7,0x43,0x96,0x5d,0x1c,0xb3,0xfc,0x97,0xf2,0xf1,0xe1,0xf5,0x9a,0xce,0xc7,0x7d,0xcf,0x2d,0xcc,
        0x5f,0x98,0x8f,0xb6,0x3c,0xbe,0x2b,0xab,0x58,0x03,0xd3,0x56,0x2f,0xc7,0xea,0x79,0xfd,0xf5,0x43,0xc3,0xb2,0xde,0x34,0xc7,
        0x96,0x4b,0x77,0x7a,0x87,0xb5,0xa9,0xaa,0xac,0x46,0xe4,0x3b,0x6a,0xbb,0x91,0xe5,0xdb,0x96,0xef,0x57,0xc7,0xf7,0x6d,0xe3,
        0x21,0x7c,0x2f,0x28,0xff,0x4a,0xb1,0x2c,0x6b,0x58,0xe7,0xf5,0x79,0xfe,0x71,0xdd,0x87,0xe0,0x75,0xf8,0xde,0x43,0x47,0x7d,
        0xfe,0x5c,0x37,0xff,0x98,0x8e,0x6f,0x41,0xdc,0x9f,0xe9,0xce,0xdf,0xef,0xff,0x97,0x60,0xfb,0x77,0xf9,0x0b,0xf3,0xff,0x56,
        0x1f,0xaf,0xf5,0x5b,0x2d,0xfb,0x25,0x2f,0x65,0x17,0x1d,0x6d,0x26,0x7c,0x6c,0x6b,0x33,0x5d,0xaf,0xdf,0x34,0x9f,0xd3,0xf4,
        0xb5,0xc5,0xee,0xf5,0xba,0xc7,0xcd,0xbc,0xe5,0xf9,0xfb,0xf4,0x34,0xa5,0xeb,0xfd,0x38,0xf0,0xf5,0x5b,0xb0,0xce,0x8f,0x16,
        0xc5,0x7b,0xf7,0x17,0x1f,0xc6,0xa8,0xf0,0xfa,0x4b,0xec,0xe3,0xa0,0x79,0xe5,0x8c,0x5e,0x8f,0xeb,0x79,0xed,0x2f,0x6e,0x1e,
        0x10,0xbb,0xde,0x47,0xf1,0x31,0xa2,0xb7,0x7d,0xbb,0xd6,0x95,0xce,0xd3,0xe1,0x9c,0x86,0x43,0xc7,0xe3,0x06,0xd2,0x08,0xb0,
        0x31,0x79,0x36,0xe5,0x78,0xf2,0x89,0xfd,0x9c,0x43,0x04,0xc7,0x98,0xa9,0x8f,0xa9,0xf3,0xb5,0x89,0x32,0xdd,0x48,0x59,0x6c,
        0x25,0x1d,0xc0,0x66,0xe5,0xd9,0xe0,0xe3,0x68,0xc4,0xf9,0x79,0xfe,0xfe,0xda,0xc2,0x3c,0x12,0x97,0x55,0x43,0x7e,0xbe,0xe5,
        0x1f,0x3f,0x4b,0x5a,0x2b,0x3d,0x5b,0xb9,0xaf,0xff,0xed,0xde,0xd2,0xd4,0x69,0x99,0xb3,0xad,0xf5,0xb5,0xc3,0xb5,0xe2,0xcc,
        0x99,0xbf,0xd4,0xf5,0x03,0x8c,0xd1,0xf5,0x7f,0x20,0x75,0xf5,0xef,0xf5,0xac,0xfe,0x07,0xe5,0x3e,0x8b,0xff,0xfe,0x87,0x14,
        0xea,0xff,0x27,0xd7,0xf4,0x9d,0x05,0x5b,0x39,0xce,0xad,0x51,0x0e,0x5d,0xdf,0xbd,0x31,0xe5,0x37,0xc5,0x97,0xfe,0xff,0xef,
        0xb8,0xff,0x3f,0xe9,0x6f,0x03,0xd7,0xfc,0x7e,0xea,0xff,0xff,0xff,0x31,0x52,0xd7,0xff,0x0d,0xd6,0xff,0xcf,0xed,0x29,0x7b,
        0x3f,0xce,0x54,0xdb,0x87,0xaf,0xfb,0xae,0x21,0x85,0xfb,0xbf,0xaf,0xc5,0x08,0xff,0xaf,0xae,0xa9,0xac,0xff,0x5c,0x6a,0xfb,
        0x5f,0xbc,0xa7,0x86,0x18,0x6d,0x79,0x7a,0x1f,0xeb,0x6d,0x1e,0x55,0x7d,0x77,0x68,0x4c,0x5e,0xa6,0xfc,0xcf,0xd6,0x9c,0x1e,
        0xf2,0xc3,0x2a,0x8a,0x7b,0x1d,0xd7,0x8a,0xf5,0x99,0x0d,0xbd,0xff,0x34,0x75,0x7a,0xe1,0x1a,0xad,0x71,0xef,0xfa,0x98,0xfe,
        0x1b,0x6e,0x7f,0xfb,0xe5,0xf0,0xf5,0x7b,0xfe,0xe5,0x55,0xb5,0xce,0x63,0x6d,0xfd,0xea,0x7f,0x6f,0xfa,0x1e,0x63,0xd7,0x7f,
        0xac,0x3d,0xaf,0x6f,0xd7,0xe7,0xd8,0xe2,0x21,0x3f,0xe6,0x4d,0xeb,0x36,0xed,0xe3,0xf1,0x2d,0x6f,0x97,0x6d,0xca,0x32,0xf8,
        0x6b,0x99,0xf7,0xbe,0xc7,0xa5,0xad,0x15,0xa7,0x2f,0x7e,0x4c,0xbe,0xfb,0xde,0x2b,0x54,0x6d,0xac,0xaa,0xab,0xc7,0xf3,0xb2,
        0x8b,0xb7,0x36,0x56,0xb4,0x07,0xe5,0xbf,0x4c,0xf9,0x87,0xcb,0x1f,0x5f,0xdf,0x3b,0xbe,0x2b,0xff,0x6d,0xcf,0x09,0xfa,0xeb,
        0x63,0x5b,0xe9,0xfd,0x5c,0x94,0xfd,0x32,0x65,0x3a,0x44,0xd5,0xd7,0xc7,0x4a,0x9d,0xd7,0xd4,0xe5,0xdd,0xa7,0xef,0xfe,0xd9,
        0xd4,0xe9,0xdb,0xb8,0x3c,0x78,0x64,0x53,0xbe,0xb0,0xb8,0xd4,0x75,0xdc,0x57,0xdf,0xa9,0xd3,0x04,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xec,0xcd,0xdb,0xef,
        0xb7,0x7f,0x4e,0xa9,0xcb,0x1f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xf9,0x3c,0x6e,0x20,0x0d,0x00,0x00,0x00,0x00,
        0x8c,0x77,0xca,0xde,0xa4,0x4e,0xcb,0x5a,0xf9,0x6c,0x93,0x65,0xd9,0x2e,0xdd,0x96,0x52,0xa7,0x03,0xd8,0x97,0x62,0xdc,0xb8,
        0xcf,0xe6,0x19,0x3f,0x7e,0x8f,0x54,0xad,0x7f,0x9a,0x59,0xea,0xb2,0x9c,0xe2,0x18,0x68,0x5a,0xd6,0x66,0x2b,0x69,0x4f,0x9d,
        0x8e,0xbd,0xab,0x97,0xe3,0x6d,0x83,0x6b,0x6e,0xff,0xb0,0xa6,0xae,0xfe,0x73,0x0a,0x1e,0x63,0xb6,0xdd,0xd2,0x71,0x2c,0x1c,
        0x8b,0x9b,0xc6,0xe6,0xa5,0xc7,0xeb,0x98,0xe3,0xd4,0x71,0x60,0xbe,0x9f,0x7b,0xca,0x3f,0x75,0x9b,0x9a,0xdb,0x5d,0xe9,0xbe,
        0xd4,0xd7,0x6e,0xae,0xed,0x38,0x1c,0x3b,0xff,0xe9,0xd3,0x57,0x6e,0xd5,0xf1,0x35,0x76,0xfe,0xda,0x57,0xfe,0x53,0x0d,0x69,
        0x1f,0xf5,0xb2,0x1a,0x3a,0xff,0xde,0x4a,0x3d,0xae,0x71,0xfd,0x62,0x89,0x76,0x13,0x8e,0x3d,0x7f,0xcf,0x0f,0xab,0xab,0x5f,
        0xbf,0xda,0xda,0x79,0x41,0x8a,0xbe,0xdf,0xb6,0xbf,0xfa,0x3c,0x3a,0x7c,0xaf,0xaa,0xbf,0xb1,0xfd,0xbf,0xda,0xcf,0x1f,0x13,
        0xc5,0xc6,0x6f,0xda,0x76,0x2f,0xc7,0xbf,0xa5,0xc6,0xf0,0x30,0xff,0x31,0xed,0x3f,0xdc,0xf7,0xd0,0xfa,0x0f,0xdb,0xd5,0xd4,
        0xfa,0x67,0x3e,0x4b,0xcf,0xd1,0x9b,0x62,0x4e,0xed,0x07,0x4d,0xe3,0xd3,0xd0,0x63,0x51,0x5b,0xbc,0xb9,0xde,0x1f,0x7b,0xde,
        0x32,0xf5,0xfd,0x6b,0x74,0xed,0xe7,0x2f,0x9f,0x59,0xcc,0x9c,0x77,0xea,0xf1,0x27,0xf4,0x9f,0xe7,0xf9,0xe0,0x7f,0xad,0x48,
        0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,
        0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,
        0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,
        0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,
        0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,
        0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,
        0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,
        0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,
        0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0x4f,0x3c,0xf1,0xc4,0x13,0xef,0x5a,
        0xe2,0x9d,0xb2,0x61,0xfe,0xb9,0xa0,0x30,0xd6,0x3f,0xae,0x40,0x98,0xa7,0xff,0xcd,0x32,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x60,0xe7,0x4e,0xc7,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x30,0xd2,0x43,0x0e,0x00,0x7c,0x36,0x59,0x96,0xb5,0x3a,0x0d,0x14,0xbb,0x8f,0xbf,0x9d,0x1d,0xcb,
        0xc7,0x18,0xd9,0x80,0x75,0xeb,0x4e,0x3d,0xcb,0x4f,0x2d,0xeb,0xb6,0x6d,0xd7,0x96,0x36,0xae,0xc3,0xd0,0xf6,0x1e,0xd3,0xfe,
        0x01,0xd8,0x96,0xef,0x67,0x4f,0x1b,0x48,0xc7,0xde,0x5d,0xfb,0xb1,0xaf,0xef,0x78,0x3f,0x75,0xce,0xb0,0xf6,0x9c,0xe3,0x39,
        0x78,0x7d,0x3c,0x4e,0x72,0xf3,0xdb,0xf9,0x1c,0xe2,0x7b,0xe9,0xb7,0xd2,0x7d,0xe9,0xa9,0x5c,0xfe,0xeb,0xd9,0x29,0x6b,0x56,
        0x9c,0x07,0x4c,0xd1,0x74,0x4e,0x53,0xc5,0x1d,0xfb,0x18,0xee,0xe7,0x21,0xcf,0x5a,0x7d,0xcf,0xbf,0x9c,0x1f,0xbf,0xbc,0x3e,
        0x16,0xf9,0x79,0xcc,0x2f,0xaa,0x32,0x28,0xd6,0x79,0xca,0x8f,0xe7,0xf7,0x8f,0x1f,0x1e,0x3f,0xa6,0xfd,0xfd,0xbe,0x8b,0xfd,
        0x15,0xcb,0x5f,0xce,0x9e,0xcb,0x75,0xc2,0xf2,0x6b,0x6b,0xb3,0xa9,0xfb,0x0d,0xdb,0x55,0x8d,0x1d,0x7f,0x2a,0x1f,0xef,0xbf,
        0x9c,0xdb,0xf0,0x97,0xc3,0xab,0x97,0xb3,0xe7,0xf2,0x79,0xe1,0xe9,0xcb,0xb9,0xad,0x9e,0x55,0xaf,0xbf,0xbf,0xae,0x9b,0x95,
        0x8f,0x87,0x68,0x8f,0xe7,0x7d,0x3c,0xbe,0xee,0x27,0xfb,0xe9,0x7e,0xe0,0x3e,0xa6,0x7a,0x2a,0xf3,0xf1,0x96,0x9f,0x63,0x82,
        0xf8,0x55,0x79,0x1e,0x7f,0x96,0xed,0xf7,0x9b,0xf3,0x18,0x71,0xf6,0xbd,0xf4,0x78,0xf6,0x70,0xf6,0xd7,0x48,0x0f,0x3b,0xf4,
        0x58,0xcb,0x67,0x2a,0xf5,0xe3,0x68,0xdf,0x3a,0x73,0xbe,0x5f,0x8f,0xfb,0x5c,0xf6,0xb9,0x7a,0x7b,0x7c,0x29,0xbd,0x8e,0xef,
        0xe7,0xf5,0xdb,0x5c,0xf6,0xd1,0x7e,0x6c,0x6d,0x3a,0xc6,0xfe,0xe5,0xfc,0xfa,0x5b,0xf1,0xd9,0x5f,0x36,0xee,0x38,0xb1,0xe6,
        0xb1,0x65,0xee,0x58,0xa7,0x19,0xf7,0x3d,0x75,0xde,0x9b,0x7a,0xde,0x78,0xed,0xf3,0x76,0xe2,0xda,0xdc,0x6d,0xcd,0x35,0xb4,
        0x8d,0xbd,0xa7,0x7f,0x4f,0xc6,0x96,0xf1,0xd6,0xce,0xa1,0x3f,0xdb,0xf8,0x79,0xdc,0x40,0xb9,0xb1,0x0d,0xa9,0xc6,0x8d,0x94,
        0xf1,0x97,0xcc,0xcf,0x50,0xc7,0xc4,0x69,0x4d,0x5d,0x76,0x63,0xca,0x72,0x4b,0x69,0x4b,0x5d,0x5e,0x6d,0x9a,0xae,0x95,0x6e,
        0x29,0xfd,0x45,0xec,0xdb,0x0d,0x94,0x53,0x5f,0x1d,0xf7,0xad,0xdf,0x96,0x87,0xd4,0xe3,0xfa,0x96,0xc7,0x7f,0xa8,0xda,0xdf,
        0xef,0x91,0xf6,0xd2,0x7e,0xa5,0x6f,0xd9,0xf4,0xee,0x2d,0xfd,0x73,0xe7,0x77,0x4a,0xde,0xa7,0x96,0x57,0xdf,0xf6,0xa7,0x09,
        0xb1,0xd6,0x4e,0x1b,0xd7,0xd9,0x7f,0x00,0x0a,0x29,0xc7,0xbf,0xb9,0xc6,0xdf,0x54,0xe3,0xff,0x9e,0x8e,0x0d,0xb1,0x69,0x4d,
        0x7d,0xac,0xdd,0xa2,0xbe,0xb2,0xda,0x5b,0x5b,0x68,0x6a,0x13,0x61,0x7e,0xd7,0x98,0xfb,0x14,0xfb,0xdf,0xe2,0xf5,0xad,0x2d,
        0x99,0x63,0x6c,0x4e,0xdd,0x7f,0x96,0xea,0x7f,0x73,0xb6,0xf9,0x35,0xda,0xfb,0x98,0x3a,0x5f,0xb3,0x7f,0xb4,0x95,0x47,0xea,
        0xfa,0x4f,0x2d,0x7b,0xbd,0x87,0x38,0xa5,0x75,0xeb,0x7d,0xa9,0xfd,0xa7,0xee,0x57,0x53,0x84,0x9f,0x11,0x75,0x7d,0x5e,0x14,
        0x7e,0xa6,0xd4,0x54,0x1e,0x5d,0xaf,0xd7,0xee,0xef,0x5d,0x9f,0x7f,0xc5,0x7e,0xf6,0xda,0xd5,0x7f,0x6e,0x7b,0xf4,0x6d,0x9f,
        0xba,0xee,0x53,0x4b,0x3f,0xfe,0x6d,0x4f,0x5b,0x7b,0x6d,0x5a,0xb7,0xad,0xdd,0xfd,0xba,0xe3,0xfc,0xcf,0xdd,0xae,0xfa,0xfa,
        0xe7,0x92,0xf1,0x53,0xe5,0x79,0xed,0xf8,0x43,0xda,0xd8,0x9c,0x7d,0x34,0xf5,0xf8,0xb1,0x77,0xa9,0xfb,0xe9,0x1e,0xa4,0xae,
        0x23,0xf5,0xbf,0x5c,0x3d,0xa5,0x4e,0xf3,0x5a,0xf9,0x4f,0x9d,0xb6,0x58,0x31,0xf7,0xac,0xd5,0x4d,0xbd,0x7f,0x04,0x20,0x95,
        0xd4,0xc7,0x8b,0xd4,0xe3,0xfd,0x90,0xb4,0x56,0xc7,0x84,0xfa,0xff,0xaa,0xc5,0x1c,0xfb,0xd8,0x66,0xfb,0x8b,0xad,0xf7,0x14,
        0xe9,0x1f,0xf2,0x1d,0x24,0x73,0xe4,0x23,0x75,0xfd,0xc5,0x5c,0xd7,0xdd,0xa2,0xad,0xb5,0xc7,0x2d,0xa5,0x3f,0x26,0xfe,0xd0,
        0x76,0xbc,0x37,0x53,0xbe,0x5b,0xa8,0xeb,0x1a,0xe1,0xd2,0xf5,0x18,0x7b,0x5e,0x53,0xaf,0xb7,0x7a,0x3d,0x0e,0xad,0xfb,0xbd,
        0xd6,0xff,0xd8,0x71,0x3a,0xbc,0x0e,0x3c,0xe4,0x7f,0x62,0xb3,0x6c,0xb9,0x7a,0x8f,0x69,0x6b,0x75,0x53,0xcf,0x7f,0xb7,0x5c,
        0xff,0x63,0xfb,0xed,0x90,0x72,0x69,0xfa,0x3c,0x60,0x48,0x3f,0x4a,0x5d,0x46,0x61,0x79,0x15,0x8f,0x77,0x91,0x69,0xdf,0x7a,
        0xfd,0xaf,0xd5,0x5e,0xc2,0xfd,0xd4,0xdb,0xc2,0x6d,0xcb,0xfa,0xd5,0xf3,0xd8,0x76,0x56,0xaf,0x97,0xf0,0x3b,0x37,0xe6,0x2e,
        0x87,0x14,0xf3,0x8f,0xb5,0xfa,0xc2,0x94,0x63,0x78,0xd8,0xc7,0x53,0xcd,0x59,0xb7,0x24,0xa6,0x4e,0x9b,0xd6,0x9b,0x5a,0xdf,
        0x43,0xb6,0x6f,0xfa,0x6e,0x9a,0x2d,0x8e,0xbf,0x4b,0xb7,0xe5,0xa5,0x8f,0xff,0x55,0x8c,0x98,0xba,0x3b,0x75,0xbc,0x9e,0x5b,
        0x6c,0x3b,0xed,0x5a,0xb7,0x9e,0xd7,0xbe,0xb2,0x0f,0xcb,0xa6,0xa9,0xac,0xea,0x79,0xee,0x2a,0xdb,0xd8,0x36,0xdb,0x15,0xab,
        0xab,0x1d,0xc4,0xb6,0x13,0x68,0x92,0x7a,0x0c,0xdc,0xc2,0xb8,0x3a,0x74,0x5e,0x58,0x9f,0x47,0x0c,0x3d,0xcf,0xa8,0xca,0x3c,
        0xdc,0xbf,0xfa,0x4f,0x5b,0xd7,0x6d,0x75,0xde,0x57,0x7e,0x3e,0x7f,0x66,0xcf,0xd2,0xdf,0xff,0x3f,0xde,0xb1,0xc3,0xdb,0x77,
        0xdf,0xf6,0xbb,0x3b,0xbb,0x2d,0xdd,0x97,0xea,0x71,0x6e,0x27,0x68,0x4b,0x73,0xcc,0xf7,0xf6,0xae,0x21,0xf5,0xf8,0xbd,0x85,
        0xf1,0x3f,0xbc,0xce,0x3f,0xa4,0xff,0x4c,0x1d,0xff,0xef,0x46,0x98,0xb7,0xff,0xef,0xc7,0x94,0xe3,0xf9,0x52,0xf5,0xb7,0x77,
        0x7f,0xac,0x68,0x8e,0x98,0x63,0xeb,0xaf,0xda,0xbe,0x3e,0x7f,0x1f,0x1a,0x0b,0xae,0x4d,0xea,0x31,0x7d,0xcd,0xe3,0xfd,0xd0,
        0x63,0x3b,0xda,0xdf,0xd2,0x86,0x7c,0x7e,0xd8,0x67,0xee,0xb2,0xaf,0xaf,0x37,0x64,0x9b,0xa6,0xe7,0x6b,0x8f,0x07,0xa7,0xe0,
        0xf9,0xa9,0x61,0x9d,0x2c,0x78,0xac,0x9f,0x0b,0x74,0x7d,0xe6,0x54,0xf7,0xd9,0xe7,0x8f,0x4b,0xf7,0xff,0x3d,0x8c,0x13,0x43,
        0x3e,0x4b,0x6a,0x3b,0x2e,0xdd,0xd5,0xf6,0xb5,0x6e,0xf9,0xc7,0x5d,0x23,0x39,0x6e,0xe0,0x7a,0xcf,0x94,0xeb,0x41,0x4d,0xef,
        0xdf,0xb5,0x5c,0xdf,0xb9,0xab,0xad,0xdf,0x7f,0xfd,0x24,0xbe,0xfc,0xf6,0x50,0x86,0x63,0xca,0x36,0x46,0x78,0x4d,0xec,0xae,
        0xa6,0x5a,0x3e,0xee,0xfa,0xd5,0x54,0xe9,0xc7,0x8f,0xa5,0xa5,0x1e,0xe3,0x81,0x6d,0x8a,0x1d,0x3f,0x52,0x8f,0x61,0x43,0xe7,
        0x5d,0xe1,0x39,0x40,0xd7,0xdc,0xab,0xe9,0x7b,0x41,0x62,0xcb,0x6f,0xce,0xf3,0xb7,0x31,0xe3,0x79,0x31,0x6f,0xbc,0x4f,0x58,
        0x07,0x55,0x3a,0xe6,0xfe,0x1c,0x7e,0x48,0x1d,0xf6,0xcd,0xbd,0x19,0xd7,0xff,0x4f,0x2d,0xcf,0xb7,0x26,0xa6,0xfe,0x9b,0xae,
        0x09,0xf4,0x8d,0x0b,0x31,0x6d,0x68,0xea,0xf9,0xff,0x5e,0xca,0x3e,0xe6,0x1a,0xcb,0x98,0x6b,0xae,0xae,0x9f,0x00,0x9f,0xd5,
        0x92,0x63,0x74,0x38,0x47,0x83,0x6b,0x33,0xb5,0x7d,0xa7,0x98,0x4b,0x3d,0x07,0xe6,0xd8,0xe7,0x98,0xbc,0xfb,0x7e,0x7a,0x00,
        0x00,0x00,0xe6,0x96,0xfa,0x3a,0x01,0xdb,0xb9,0xfe,0xc1,0x7c,0x7d,0x6a,0xca,0xbd,0x0b,0xea,0x1f,0xde,0x4b,0x3d,0x1e,0x03,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2c,0x2d,0xf5,0xfd,0x9a,0x4c,0xbb,0x77,0x35,0xf5,0xf7,0xca,0x02,0x00,0x5c,
        0x9b,0xa7,0x9a,0xef,0x35,0xa9,0xd3,0x15,0x0a,0xd3,0x54,0xa5,0xf7,0xa9,0xe1,0x3d,0x60,0x3f,0xea,0x63,0x4f,0xea,0xb4,0xc0,
        0x5c,0x8a,0xdf,0x7f,0xb9,0xe9,0x50,0xb5,0xfb,0xee,0xdf,0x90,0x49,0x7f,0x0d,0x0d,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0xff,0x0f,0x3f,0x00,0xec,0x89,0xef,0x45,0xbb,0x7e,0xdd,0xdf,0x1b,0x74,0x2d,0xd2,
        0xcf,0x41,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0xc8,0x00,0x00,0xae,0xd0,0x56,0xe7,0x3f,0x45,0xfc,0xdf,0x23,
        0x85,0xe9,0xbe,0x2d,0xa5,0x2e,0xdb,0x3d,0x9b,0x52,0xfe,0xa9,0xdb,0x4f,0x15,0xff,0xd8,0xe0,0xb6,0x23,0xbd,0x5b,0x49,0xff,
        0x5c,0x9a,0xf2,0x5f,0x2f,0x87,0xa1,0xfd,0xa4,0x5a,0x37,0xb6,0xfc,0x86,0xb4,0x9f,0xbe,0xfa,0x5c,0xb2,0x9d,0x2c,0x55,0xe7,
        0xa9,0xcf,0x5f,0x01,0x00,0x60,0x88,0xd4,0xe7,0x2f,0xd7,0x2c,0x3c,0xf7,0x4a,0x5d,0xd7,0x68,0xff,0x53,0x84,0x65,0x77,0x2c,
        0x97,0x77,0x5d,0x83,0xd8,0xbb,0xae,0xb6,0x13,0xae,0x3b,0x47,0x99,0x7e,0x36,0x8f,0xf9,0xe1,0x9d,0x9b,0xb3,0x53,0x16,0xef,
        0x98,0x5f,0x0c,0xd9,0xa6,0xee,0x21,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfa,0xf8,0xfe,0x7f,0x00,0x00,0xa0,0x92,0x01,0x00,0xb3,0x19,0xfb,0x7b,0x9c,0x6b,0x8b,
        0x99,0x23,0xa4,0x4e,0x23,0x4b,0x4a,0xff,0x1b,0xb8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x43,0x64,0xf9,0xe1,
        0x9d,0x53,0x76,0xf8,0xb0,0xac,0xed,0xfd,0xe2,0xf9,0x43,0x0e,0x90,0x5e,0x31,0x1e,0xed,0xcd,0x5f,0xf3,0x37,0x73,0xef,0x2f,
        0x66,0x9f,0xa9,0xeb,0x0c,0x80,0x7d,0xf8,0x78,0x0c,0x49,0x7f,0x0e,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8c,0x93,
        0xb1,0x1b,0x73,0xd5,0xdf,0xd0,0xed,0xe7,0x68,0x3f,0xf5,0xf5,0x8f,0xa5,0xdb,0xda,0xf3,0x4a,0xea,0xfe,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0xb4,0x63,0x76,0xd8,0xa5,0xea,0xf7,0x0b,0xd7,
        0x88,0xd1,0x25,0xe3,0xaa,0x0c,0xff,0xcd,0xe6,0xf4,0x7d,0x78,0x8a,0xd4,0xe5,0x0d,0xcc,0x2f,0x76,0xfc,0x4a,0x9d,0xce,0x98,
        0x3c,0xa4,0x4e,0xc7,0xda,0x9a,0xe6,0x22,0x7d,0xef,0xcf,0x21,0x66,0xff,0x5b,0x29,0x93,0xd4,0x75,0x94,0xc2,0xf0,0xb9,0x49,
        0x4c,0xff,0x4f,0xff,0xbd,0xb6,0x5b,0x50,0xff,0x0e,0xda,0xbe,0xef,0xbc,0x85,0xb9,0xa4,0x1e,0x23,0x52,0x8f,0x69,0xd7,0x6c,
        0x2b,0xe5,0x7f,0xcd,0xe7,0xaf,0x4c,0x1d,0x7f,0x98,0x9b,0x72,0x8d,0x2f,0x9f,0x7a,0x3b,0x0c,0xdb,0x64,0x5b,0x1b,0x1d,0x53,
        0xbe,0xa7,0xc8,0x6d,0xd5,0x5d,0x7b,0x39,0xc4,0x8e,0x19,0x5b,0x2c,0xe3,0x21,0xed,0x28,0x6c,0x8f,0x5b,0xcc,0x0f,0xc0,0xb5,
        0x32,0x3f,0x05,0xb6,0xa6,0x1a,0x97,0x7e,0xbf,0x12,0xa9,0xaf,0x91,0x01,0x00,0xeb,0x4b,0x7d,0x1f,0xef,0x92,0xae,0x35,0xaf,
        0x31,0x75,0x3a,0xa5,0x3d,0x54,0xcf,0x6f,0x03,0xc5,0xb2,0x29,0xf7,0x9a,0x34,0xe5,0x65,0xea,0x3d,0xdc,0xa9,0x3f,0x3f,0x03,
        0x00,0x00,0x00,0xd8,0xba,0x2d,0x7c,0x96,0x0a,0xa4,0x71,0xec,0xd0,0xf7,0x7e,0xac,0xa1,0xb1,0xda,0xd6,0x4d,0x5d,0x56,0x70,
        0x6d,0x52,0xdf,0x4f,0x35,0x75,0xde,0xd0,0x35,0xa6,0xcc,0x79,0x1f,0x57,0x5b,0xb9,0xa5,0xae,0x3f,0x80,0x54,0x52,0x1d,0x03,
        0xe6,0x9a,0x0f,0xa6,0x3e,0x96,0x8d,0xc9,0x77,0xdd,0xd8,0xe3,0x57,0xe1,0x65,0xa6,0x32,0x04,0x00,0x00,0x00,0x00,0x9a,0xf5,
        0x7d,0x46,0xd5,0xb5,0xac,0x6f,0xfb,0x98,0xcf,0xd1,0x52,0x5f,0xcf,0x84,0xad,0x89,0xe9,0x9b,0x95,0x98,0xfb,0x5e,0xb6,0x2c,
        0x75,0x59,0x6f,0x4d,0xdf,0xfd,0x04,0x61,0xf9,0x8d,0xfd,0xfc,0xe5,0xa9,0x94,0xba,0xfe,0x01,0x00,0x80,0x7d,0x49,0x7d,0xce,
        0xb4,0xa6,0xe7,0x9e,0xd7,0x4d,0xe7,0x68,0x68,0x3f,0xda,0x08,0x7c,0xd4,0xd4,0x2f,0xd6,0xee,0x87,0xe1,0xf3,0x53,0xf0,0x7c,
        0xcc,0xbe,0x9a,0xf6,0xd9,0xd5,0xef,0x9b,0xde,0x6b,0xdb,0x7e,0x4c,0xfa,0x63,0xf6,0xd7,0xb4,0xfd,0x90,0xba,0x89,0xdd,0x66,
        0xae,0xb2,0x9d,0x6b,0xdc,0x8d,0x89,0xb3,0x64,0xfc,0x3d,0xbb,0xd6,0x7c,0x01,0xcb,0x7b,0xc8,0x0e,0xf9,0x3b,0x79,0xf0,0x1a,
        0x68,0x76,0x05,0x7d,0xe5,0xdb,0xf1,0xf0,0x35,0x5a,0x76,0x31,0x7a,0xfb,0x19,0x95,0xdf,0x51,0x9d,0x2c,0x7e,0xa2,0xfc,0xbe,
        0xe6,0x39,0x28,0xff,0xca,0xd0,0x7d,0x8e,0xdd,0xee,0x22,0xaf,0x6d,0x5b,0x3c,0x8f,0x31,0x53,0x59,0xa4,0xee,0x33,0x6b,0xf8,
        0x9f,0xec,0x70,0xb3,0x86,0x87,0xfc,0xcb,0xd7,0xb3,0xf1,0x63,0x59,0x7e,0xae,0x8f,0x38,0x43,0xd6,0x1d,0xb3,0xcf,0x3c,0x42,
        0xd3,0x7c,0xa7,0x3f,0x4e,0x77,0xfe,0xa7,0x1d,0x07,0xda,0xd2,0xba,0x81,0x36,0xc8,0x27,0x10,0xd7,0x6f,0x96,0xb5,0x44,0x9a,
        0xb7,0x92,0xb7,0x2d,0xea,0x29,0x9b,0x5f,0x16,0xac,0xcb,0xac,0xb4,0x64,0xfb,0xcb,0x4a,0x6d,0xef,0xb5,0xe5,0xb9,0xd2,0x54,
        0x0e,0xaf,0xef,0x65,0x17,0x9d,0x69,0x4f,0xd1,0xff,0xc2,0xfd,0xb5,0xec,0xff,0x3f,0xf2,0xc3,0x4d,0x28,0x79,0x5b,0x4c,0xa9,
        0x56,0x5e,0x4d,0x75,0x1f,0x6b,0x89,0xb4,0x0d,0x89,0x3f,0x24,0x7d,0x4d,0xe9,0x0d,0xd7,0x8d,0xcd,0xdb,0x2f,0x1d,0xfb,0x8c,
        0x95,0xb5,0x3c,0xb6,0xc5,0x6a,0xdb,0xb6,0x6d,0x3b,0x1a,0xd4,0xc6,0x86,0xac,0xc3,0xd4,0x38,0x5d,0xfb,0x9e,0x3b,0xee,0xe0,
        0x38,0xfd,0xbf,0x85,0x95,0x67,0xe1,0xf1,0xaa,0xa9,0x1c,0xd7,0xaa,0xab,0x48,0x1f,0x8e,0x51,0xd5,0xf2,0x11,0xfb,0x8a,0x8d,
        0xd7,0x14,0x6b,0x74,0xbc,0x85,0xd2,0xb9,0x96,0xb0,0xdd,0x34,0xb6,0xa3,0x04,0xf1,0x57,0x69,0xab,0x57,0xa2,0xeb,0xd8,0x19,
        0x4a,0xfd,0xfb,0x8d,0x6d,0x7e,0xd4,0x3c,0xd6,0x84,0xeb,0xa5,0x2e,0xeb,0xb0,0x7c,0x97,0xaa,0xbb,0xc1,0x31,0xdb,0xc6,0xbb,
        0xf0,0xbd,0xa6,0x6d,0xfa,0xf6,0x13,0x11,0xa7,0x73,0xbb,0x98,0x75,0xba,0xbc,0x1f,0x2f,0x92,0xb7,0x55,0xe6,0x33,0xe2,0x78,
        0x91,0x3c,0xcd,0xd7,0x66,0x54,0x9f,0x8e,0xd5,0xb4,0xfd,0xa4,0xf9,0xc2,0xb0,0xfb,0xa7,0x1e,0xb2,0x2c,0xaf,0x9b,0xe3,0x9e,
        0xac,0x2d,0x0b,0xf3,0x3b,0x36,0xff,0x53,0xb7,0xdf,0x7b,0xfe,0xd7,0x4e,0x17,0x71,0x86,0x8e,0x6d,0x6d,0xd7,0x47,0x63,0xe6,
        0x63,0x43,0xae,0x9d,0x0d,0x99,0xbf,0x45,0x5f,0x03,0x2b,0xd3,0x1f,0x35,0x3f,0xcc,0xb6,0x35,0x37,0x1e,0x3b,0x17,0x4e,0x9d,
        0x86,0x54,0xf9,0x0e,0xaf,0x65,0xce,0x59,0x1e,0x5d,0xd7,0x4b,0x53,0x5e,0x33,0x1e,0x23,0x8b,0xb8,0x66,0x41,0xbc,0xa5,0xaf,
        0xef,0x44,0xcd,0x1f,0xa7,0xbe,0xdf,0xb4,0xde,0x90,0xf8,0x4b,0x1b,0x1b,0xbf,0x27,0xfd,0xef,0xfa,0xe7,0x94,0xf2,0x1f,0x52,
        0x96,0x63,0xb7,0x2b,0x75,0x8e,0x31,0x11,0xe9,0x4f,0x3d,0xbf,0xe4,0x7a,0x9c,0x36,0x90,0x86,0xa5,0xf3,0xb7,0x35,0x53,0x7e,
        0xff,0xe3,0x72,0xff,0x7f,0xfa,0x73,0x10,0x00,0x00,0x00,0x60,0x19,0x3f,0xcf,0xff,0xe7,0xff,0x9f,0x3c,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xa7,0x4e,0x59,0xf6,0x2a,0x75,0x3a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xed,0x38,0x65,0x19,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0x6e,0x1c,0xd3,0xc6,0xbe,0x21,0x9d,0xd4,0x6d,0x8f,0xb4,0x8e,0x47,0x60,0xaf,0x52,0x8f,
        0x1f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xdf,0xee,0x1c,0x1a,0x01,0x00,0x83,0x40,0x00,0x7b,
        0x2a,0xba,0xff,0xc8,0x15,0x0c,0x51,0xee,0x88,0x88,0x0f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0xc0,0x29,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x80,0x96,0xb4,0xdf,0x0f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x8e,0xe4,0x3e,
        0x43,0x2a,0x52,0xa2,
    };
};

class BreakPairTable
{
public:
    BreakPairTable() = delete;

    static BreakOpportunity getLineBreakOpportunity (LineBreakType a, LineBreakType b)
    {
        return (BreakOpportunity) table[(int) a][(int) b];
    }

private:
    static constexpr uint8_t table[44][44]
    {
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x2,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x2,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x0,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x1,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x0,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x1,0x1,0x0,0x1,0x1,0x2,0x1,0x0,0x0,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x2,0x0,0x4,0x0,0x2,0x0,0x0,0x0,0x2,0x2,0x2,0x0,0x2,0x2,0x2,0x0,0x0,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x0,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
    };
};
}
