with TOML;

with Alire.Utils.Switches;
with Alire.Properties.Configurations;
limited with Alire.Roots;

private with Ada.Strings.Unbounded;
private with Ada.Containers.Hashed_Maps;
private with Ada.Strings.Unbounded.Hash;
private with Ada.Containers.Indefinite_Holders;
private with Ada.Containers.Indefinite_Ordered_Maps;

package Alire.Crate_Configuration is

   Default_Root_Build_Profile : constant Utils.Switches.Profile_Kind :=
     Utils.Switches.Development;

   Default_Deps_Build_Profile : constant Utils.Switches.Profile_Kind :=
     Utils.Switches.Release;

   Root_Build_Profile : Utils.Switches.Profile_Kind :=
     Default_Root_Build_Profile;

   type Global_Config is tagged limited private;

   procedure Load (This : in out Global_Config;
                   Root : in out Alire.Roots.Root);

   procedure Generate_Config_Files (This : Global_Config;
                                    Root : in out Alire.Roots.Root);

   procedure Set_Last_Build_Profile (P : Utils.Switches.Profile_Kind);
   --  Record in local user configuration the last profile used in crate
   --  configuration.

   function Last_Build_Profile return Utils.Switches.Profile_Kind;
   --  Get the last profile used from user configuration

private

   use Alire.Properties.Configurations;
   use type Alire.Utils.Switches.Profile_Kind;
   use type Alire.Utils.Switches.Switch_List;

   package Config_Type_Definition_Holder
   is new Ada.Containers.Indefinite_Holders (Config_Type_Definition);

   type Config_Setting is record
      Type_Def  : Config_Type_Definition_Holder.Holder;
      Value     : TOML.TOML_Value;
      Set_By    : Ada.Strings.Unbounded.Unbounded_String;
   end record;

   package Config_Maps is new Ada.Containers.Hashed_Maps
     (Key_Type        => Ada.Strings.Unbounded.Unbounded_String,
      Element_Type    => Config_Setting,
      Hash            => Ada.Strings.Unbounded.Hash,
      Equivalent_Keys => Ada.Strings.Unbounded."=");

   package Profile_Maps
   is new Ada.Containers.Indefinite_Ordered_Maps
     (Crate_Name, Alire.Utils.Switches.Profile_Kind);

   package Switches_Maps
   is new Ada.Containers.Indefinite_Ordered_Maps
     (Crate_Name, Alire.Utils.Switches.Switch_List);

   type Global_Config is tagged limited record
      Map : Config_Maps.Map;

      Profile_Map  : Profile_Maps.Map;
      Switches_Map : Switches_Maps.Map;
   end record;

   procedure Use_Default_Values (Conf : in out Global_Config);
   --  Use default value for unset variable, raise Checked_Error if a variable
   --  has no default value.

   procedure Add_Definition (This     : in out Global_Config;
                             Crate    : Crate_Name;
                             Type_Def : Config_Type_Definition);

   procedure Load_Definitions (This  : in out Global_Config;
                               Root  : in out Roots.Root;
                               Crate : Crate_Name);

   procedure Set_Value (This  : in out Global_Config;
                        Crate : Crate_Name;
                        Val   : Assignment);

   procedure Load_Settings (This  : in out Global_Config;
                            Root  : in out Roots.Root;
                            Crate : Crate_Name);

   procedure Generate_Ada_Config (This     : Global_Config;
                                  Crate    : Crate_Name;
                                  Filepath : Absolute_Path;
                                  Version  : String);

   procedure Generate_GPR_Config (This        : Global_Config;
                                  Crate       : Crate_Name;
                                  Filepath    : Absolute_Path;
                                  Withs       : AAA.Strings.Set;
                                  Version     : String);

   procedure Generate_C_Config (This     : Global_Config;
                                Crate    : Crate_Name;
                                Filepath : Absolute_Path;
                                Version  : String);

end Alire.Crate_Configuration;
